<?php
/*
	class.tinytemplate.php
.---------------------------------------------------------------------------.
|  Software: tinytemplate - Simple PHP Template Class                       |
|   Version: 1.4                                                            |
|      Date: 16.08.2013                                                     |
|      Site: http://jspit.de/?page=template                                 |
| ------------------------------------------------------------------------- |
| Copyright © 2013, Peter Junk (alias jspit). All Rights Reserved.          |
| ------------------------------------------------------------------------- |
|   License: Distributed under the Lesser General Public License (LGPL)     |
|            http://www.gnu.org/copyleft/lesser.html                        |
| This program is distributed in the hope that it will be useful - WITHOUT  |
| ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or     |
| FITNESS FOR A PARTICULAR PURPOSE.                                         |
'---------------------------------------------------------------------------'
 */
 
class TinyTemplate {

  const TplStartSeparator = '{{';
  const TplStopSeparator = '}}';
  const HtmlStartSeparator = '<!--:';
  const HtmlStopSeparator = ':-->';
  const CssJsStartSeparator = '/*:';
  const CssJsStopSeparator = ':*/';
  
  //max. Verschachlungtiefe für Platzhalter
  const DeepLimit = 5;  
  
  private $tpl = false;  //Template Content
  private $assoc = array();  //Replacements
  
  private $cacheFileName = '';
  private $CacheExpired = true;
    
  /*
   * mit dem Kontruktor ein tpl-File übergeben
   */
  function __construct($tplFileName) {
    $this->tpl = @file_get_contents($tplFileName);
    if( !$this->tpl ) throw new Exception("Open Error Template-File");
    
    $keyArr = $this->PlaceHolderKeys($this->tpl);
    if ($keyArr !== false) {
      $this->assoc = $keyArr;
    }
  }

  /*
   * assign($key_or_array, $content)
   * füllt einen Platzhalter mit Inhalt
   * return false wenn $key nicht existiert, true wenn ok
   */
  public function assign($key_or_array, $content = '') {
    if(is_array($key_or_array)) {
      foreach($key_or_array as $key => $newContent) {
        if( !$this->assign($key,$newContent) ) return false;
      }
    }
    elseif(key_exists($key_or_array, $this->assoc)) {
      $this->assoc[$key_or_array] = $content;
      $keyArr = $this->PlaceHolderKeys($content);
      if ($keyArr !== false) {
        $this->assoc = array_merge($keyArr, $this->assoc);
      }
      return true;
    }
    return false;
  }
  
  /*
   * getTemplate liefert das reine Template ohne ein Ersetzen der Container
   */
  public function getTemplate() {
    
    return $this -> tpl;
  }
  
  
  /*
   * Liefert geparsten Code vom gesamten Template, nutzbar für CSS, JS + HTML
   * $noparse : ein Array von Schlüsseln, die nicht geparst werden sollen
   */
  public function renderCode(array $noparse = array()) {
  
    //check Cache
    $cacheFileName = $this -> cacheFileName;
    if( $this -> CacheExpired or !is_readable($cacheFileName)) {
      //update
      $code = $this->removeSectionsComment( $this -> renderCodePar(null, $noparse) );
      //Caching
      if( !empty($cacheFileName) and ( is_writable($cacheFileName) or !file_exists($cacheFileName) ) ) {
        file_put_contents($cacheFileName, $code);
      }
    }
    else {
      //Cache
      $code = file_get_contents($cacheFileName);
    }
    return $code;  
  }

  /*
   * Liefert den Code der Sektion von inklusive Start bis vor exklusive End mit allen Ersetzungen
   * ist End = Null, so wird die nächste HTML-Section als Ende genommen, so vorhanden 
   * ist End = '*' so wird bis zum Ende geliefert
   * $noparse : ein Array von Schlüsseln, die nicht geparst werden sollen
   */
  public function renderSection($Start = NULL,$End = NULL,array $noparse = array()) {

    return $this -> renderCodePar($this -> getSection($Start, $End), $noparse);

  }  
  
  /*
   * liefert den ungeparsten Template-Code von inklusive Start bis vor exklusive End
   * ist End = Null, so wird die nächste Section als Ende genommen, so vorhanden 
   * ist End = '*' so wird bis zum Ende geliefert
   * Nur sinnvoll für Sektionen aus der Templatedatei der Instanz
   */
  public function getSection($Start = NULL,$End = NULL) {
     if( $Start === '*') $Start = Null;
    
    $startPos = 0;
    $endPos = strlen($this->tpl);
    if( !empty($Start)) {
      $startPos = $this->posSection($Start);
      if($startPos === false) return false;
      if( empty($End)) {
        $posNext = $this->posSection('\w+',$startPos+1, false);  //not quote
        if($posNext !== false) $endPos = $posNext;
      }
    }
   
    if( !empty($End) && $End !== '*') {
      $endPos = $this->posSection($End);
      if($endPos === false) return false;
    }
    if($endPos <= $startPos) return false;
    return $this->removeSectionsComment( substr($this->tpl, $startPos,$endPos-$startPos)) ;
  }
  
  /*
   * Aktiviert das Caching und liefert true wenn der Cache nicht mehr aktuell ist
   */
  public function isCacheExpired($cacheFileName, $CacheTimeInterval = 0) {
    $this -> cacheFileName = $cacheFileName;
    $this -> CacheExpired = true;
    if(is_readable($cacheFileName)) {
      $cacheTime = filemtime($cacheFileName);
      if(is_numeric(trim($CacheTimeInterval))) {
        $CacheTimeInterval = (int)$CacheTimeInterval . ' Seconds'; 
      }
      $this -> CacheExpired = ( strtotime('+'. trim($CacheTimeInterval,' +-') , $cacheTime) <= time() );     
    }
    return $this -> CacheExpired;
  }
 
  /*
   * Entschärft Platzhalter, damit sie unverfälscht dargestellt werden können
   * '{{' -> '&#x7b{'   
   */
  public function escape($str) {
    return str_replace('{{','&#x7b{',$str);
  }  
  
  
  /*
   * private
   */
      
  /*
   * Liefert geparsten Code aus $code,
   * der Bestandteil des Templates sein muss
   */
  private function renderCodePar($code = NULL, array $noparse = array()) {
    
    if(empty($code)) $code = $this -> tpl;
     
    //noparse
    $repl = array_diff_key($this->assoc,array_flip($noparse));

    $tplPattern = array_map(
        "self::wrapTpl",
        array_keys($repl)
    );
    $maxZyk = self::DeepLimit;
    $code1 = $code;
    while(--$maxZyk) {
      $code = preg_replace($tplPattern,$repl,$code1);
      if($code == $code1) break;
      $code1 = $code;
    }
    return $code;  
  }
  
  //Packt $p in TplStartSeparator + TplStopSeparator
  private function wrapTpl($p){
    $regEx = '~' . preg_quote(self::TplStartSeparator)
      . ' *\b' . preg_quote($p). '\b[^}]*'
      . preg_quote(self::TplStopSeparator) . '~su';
    return $regEx;
  }
 
  //Liefert ein array mit Schlüsseln von Platzhaltern für Container
  private function PlaceHolderKeys($tplStr) {

    $regEx = '/'.preg_quote(self::TplStartSeparator).'([^}]+)'.preg_quote(self::TplStopSeparator).'/su';
  
    if( preg_match_all($regEx,$tplStr,$match)) {
      //return array_fill_keys($match[1],'');
      $assoc = array();
      foreach( $match[1] as $content ) {
        //content: " name | 'defaultvalue' "
        $contArr = explode('|',$content);
        $key = trim($contArr[0]);
        if($key != ''){
          $assoc[$key] = isset($contArr[1]) ? trim($contArr[1],'\' "') : '';
        }
      }
      return $assoc;
    }
    
    return false;
  }
  
  //Helper Function  
  private function regExSection($search, $option = '') {
    $regEx = '~'.preg_quote(self::HtmlStartSeparator,'~').' +?'.$search.' +?'.preg_quote(self::HtmlStopSeparator,'~').$option
    . '|'. preg_quote(self::CssJsStartSeparator,'~').' +?'.$search.' +?'.preg_quote(self::CssJsStopSeparator,'~').$option.'~su';
    return $regEx;
  }
  
  //Liefert die pos (int) von der Sektion im Template, false bei Fehler/nicht gefunden
  private function posSection($search, $offset=0, $quote = true) {
    if ($quote) $search = preg_quote($search,'~');
    $r = preg_match($this->regExSection($search), $this->tpl, $match, PREG_OFFSET_CAPTURE, $offset);
    return ($r == 0) ? false : $match[0][1];
  }
  
  private function removeSectionsComment($str) {
    return preg_replace($this->regExSection('\w+','\R?'), '',$str);
  }

}
?>